/*! Photo Feed- Extension for Flow Gallery jQuery plugin - 1.0.1
 * Copyright 2015, Nilok Bose
 * http://codecanyon.net/user/cosmocoder
*/

// polyfill for bind()
if (!Function.prototype.bind) {
    Function.prototype.bind = function(oThis) {
        if (typeof this !== "function") {
            // closest thing possible to the ECMAScript 5
            // internal IsCallable function
            throw new TypeError("Function.prototype.bind - what is trying to be bound is not callable");
        }

        var aArgs = Array.prototype.slice.call(arguments, 1),
            fToBind = this,
            fNOP = function() {},
            fBound = function() {
                return fToBind.apply(this instanceof fNOP && oThis ? this : oThis,
                    aArgs.concat(Array.prototype.slice.call(arguments)));
            };

        fNOP.prototype = this.prototype;
        fBound.prototype = new fNOP();

        return fBound;
    };
}



// Object.create support test, and fallback for browsers without it
if ( typeof Object.create !== 'function' ) {
    Object.create = function(o) {
        function F() {}
        F.prototype = o;
        return new F();
    };
}



(function($, document, window) {
    'use strict';

    var FlowPhotoFeed = {
        init: function(options, elem) {
            this.$elem = $(elem);
            this.options = options;

            var self = this, configReq;

            if( this.options.feed === 'flickr' ) {
                if( this.options.flickrOptions.sourceType === 'collection' ) {
                    configReq = this.loadFlickrCollection();
                }
                else {
                    configReq = this.loadFlickrData();
                }
            }
            else {
                configReq = this.loadPicasaData();
            }


            configReq.done(function(data) {
                var flowOptions = self.options;
                flowOptions.configData = data;
                flowOptions.configUrl = '';
                flowOptions.enableCache = false;

                self.$elem.flowGallery( flowOptions );
                self.$elem.find('div.flow-menu').find('a[data-type="audio"]').hide();
            });
        },



        // get the cached gallery data
        getCache: function(cacheId) {
            var req = $.ajax({
                type: 'post',
                data: {interval: this.options.cacheInterval, cacheFile: cacheId+'-cache.json'},
                url: this.options.cacheFolder+'/get-cache.php',
                dataType: 'json',
                global: false
            });

            return req;
        },



        // update the cache file with the fresh gallery data
        updateCache: function(json, cacheId) {
            $.ajax({
                type: 'post',
                data: {config: JSON.stringify(json), cacheFile: cacheId+'-cache.json'},
                url: this.options.cacheFolder+'/update-cache.php',
                dataType: 'json',
                global: false
            });
        },



        // function to load data from Flickr
        loadFlickrData: function() {
            var self = this,
                options = self.options.flickrOptions,
                thumbSize = 'url_'+options.thumbSize,
                imageSize = 'url_'+options.imageSize,
                cacheId = '',
                cacheReq,
                flickrUrl = 'https://api.flickr.com/services/rest/?method=flickr.photos.search&api_key='+options.apiKey+'&per_page='+options.limit+'&sort='+options.sort+'&media='+options.media+'&extras=url_t,url_m,url_o,url_s,url_q,url_l,url_z,description,media,date_upload',
                configReq = $.Deferred();

            if( options.sourceType === 'user' ) {
                flickrUrl += '&user_id=' + options.sourceId;
                cacheId = 'flickr-user-' + options.sourceId;
            }
            else if( options.sourceType === 'group' ) {
                flickrUrl += '&group_id=' + options.sourceId;
                cacheId = 'flickr-group-' + options.sourceId;
            }
            else if( options.sourceType === 'set' ) {
                flickrUrl = 'https://api.flickr.com/services/rest/?&method=flickr.photosets.getPhotos&api_key=' + options.apiKey + '&photoset_id=' + options.sourceId + '&per_page=' + options.limit + '&sort=' + options.sort+'&media='+options.media+'&extras=url_t,url_m,url_o,url_s,url_q,url_l,url_z,description,media,date_upload';
                cacheId = 'flickr-set-' + options.sourceId;
            }
            else if( options.sourceType === 'tags' ) {
                flickrUrl += '&tags=' + options.sourceId;
                cacheId = 'flickr-tags-' + options.sourceId;
            }
            else if( options.sourceType === 'text' ) {
                flickrUrl += '&text=' + options.sourceId;
                cacheId = 'flickr-text-' + options.sourceId;
            }

            flickrUrl += '&format=json&jsoncallback=?';


            // check the cache first
            if( self.options.enableCache ) {
                cacheReq = self.getCache(cacheId);
            }
            else {
                cacheReq = $.Deferred();
                cacheReq.resolve({expired: true});
            }

            cacheReq.done(function(cache){
                if( !cache.expired ) {
                    configReq.resolve(cache);  // initialize the gallery from cached data
                }
                else {  // cache expired, load fresh data
                    $.ajax({
                        url: flickrUrl,
                        dataType: 'jsonp',
                        success: function(data) {
                            var config = [];

                            if( data.stat === 'ok' && data.photos.total !== '0' ) {
                                config = self.createFlickrSetData(data.photos.photo, imageSize, thumbSize);
                                self.options.enableCache && self.updateCache(config, cacheId);  // update the cache file with the loaded data
                                configReq.resolve(config);
                            }
                            else if( data.photos.total === '0' ) {
                                self.fgLog( 'Flickr: No photos found' );
                            }
                            else {
                                self.fgLog( 'Flickr: '+ data.message );
                            }
                        }
                    });
                }
            });

            return configReq;
        },



        // load a Flickr collection that has multiple sets
        loadFlickrCollection: function() {
            var self = this,
                options = self.options.flickrOptions,
                thumbSize = 'url_'+options.thumbSize,
                imageSize = 'url_'+options.imageSize,
                cacheId = 'flickr-collection' + options.sourceId + '-user-' + options.userId,
                collectionUrl = 'https://api.flickr.com/services/rest/?method=flickr.collections.getTree&api_key='+options.apiKey,
                setUrl = 'https://api.flickr.com/services/rest/?&method=flickr.photosets.getPhotos&api_key=' + options.apiKey + '&per_page=' + options.limit + '&sort=' + options.sort+'&media='+options.media+'&extras=url_t,url_m,url_o,url_s,url_q,url_l,url_z,description,media,date_upload&format=json&jsoncallback=?',
                albumnum = options.numberOfAlbums === 'all' ? '' : options.numberOfAlbums,
                configReq = $.Deferred(),
                cacheReq;

            if( options.sourceId !== '' ) {
                collectionUrl += '&collection_id='+options.sourceId;
            }

            if( options.userId !== '' ) {
                collectionUrl += '&user_id='+options.userId;
            }

            collectionUrl += '&format=json&jsoncallback=?';

            // check the cache first
            if( self.options.enableCache ) {
                cacheReq = self.getCache(cacheId);
            }
            else {
                cacheReq = $.Deferred();
                cacheReq.resolve({expired: true});
            }

            cacheReq.done(function(cache) {
                if( !cache.expired ) {
                    configReq.resolve(cache);  // initialize the gallery from cached data
                }
                else {  // cache expired, load fresh data
                    $.ajax({
                        url: collectionUrl,
                        dataType: 'jsonp',
                        success: function(collectionData) {
                            var getSets = [],
                                config = {albums: []};

                            if( collectionData.stat === 'ok' ) {
                                $.each(collectionData.collections.collection[0].set, function(index){
                                    var setId = this.id,
                                        setTitle = this.title,
                                        request = $.ajax({
                                            url: setUrl + '&photoset_id='+setId,
                                            dataType: 'jsonp',
                                            success: function(data) {
                                                var obj = {title: setTitle};

                                                if( data.stat === 'ok' ) {
                                                    obj.items = self.createFlickrSetData(data.photoset.photo, imageSize, thumbSize);
                                                    config.albums[index] = obj;
                                                }
                                                else {
                                                    self.fgLog( 'Flickr: '+ data.message );
                                                }
                                            }
                                        });

                                    getSets.push(request);

                                    if( index === albumnum - 1 ) {
                                        return false;
                                    }
                                });
                            }
                            else {
                                self.fgLog( 'Flickr: '+ collectionData.message );
                            }

                            $.when.apply($, getSets).done(function() {
                                self.options.enableCache && self.updateCache(config, cacheId);  // update the cache file with the loaded data
                                configReq.resolve(config);
                            });
                        }
                    });
                }
            });

            return configReq;
        },



        // generate json from Flickr api containing data for a particular set or a specific user/group/tag/text
        createFlickrSetData: function(photos, imageSize, thumbSize) {
            var largePhoto, data = [];

            $.each(photos, function(i, item) {
                var obj = {};

                largePhoto = item[imageSize] ? item[imageSize] : item['url_o'];
                largePhoto = largePhoto ? largePhoto : item['url_z'];
                largePhoto = largePhoto ? largePhoto : item['url_m'];

                obj.type = item['media'];
                obj.sort = {date: parseInt(item['dateupload'], 10)};
                obj.thumbnail = item[thumbSize];

                if( item['media'] === 'photo' ) {
                    obj.source = largePhoto;
                }
                else {
                    obj.mp4 = 'https://www.flickr.com/photos/'+item['owner']+'/'+item['id']+'/play/site/'+item['secret']+'/';
                    obj.mobileMp4 = 'https://www.flickr.com/photos/'+item['owner']+'/'+item['id']+'/play/mobile/'+item['secret']+'/';
                }

                obj.title = item.title;
                obj.description = item.description._content;

                data.push(obj);
            });

            return data;
        },



         // load data from Picasa
        loadPicasaData: function() {
            var self = this,
                options = self.options.picasaOptions,
                thumbSize = '',
                imageSize = '',
                cacheId = '',
                picasaUrl = 'https://picasaweb.google.com/data/feed/api/',
                albumnum = options.numberOfAlbums === 'all' ? 1000 : options.numberOfAlbums,
                configReq = $.Deferred(),
                cacheReq;

            if( options.sourceType === 'search' ) {
                picasaUrl += 'all?q=' + options.search + '&callback=?';
                cacheId = 'picasa-search-' + options.search;
            }
            else if( options.sourceType === 'user' ) {
                picasaUrl += 'user/'+ options.username + '?&callback=?';
                cacheId = 'picasa-user-' + options.username;
            }
            else if( options.sourceType === 'album' ) {
                picasaUrl += 'user/'+ options.username + '/album/'+ options.album +'?&callback=?';
                cacheId = 'picasa-album-' + options.album + '-user-' + options.username;
            }
            else if( options.sourceType === 'collection' ) {
                picasaUrl += 'user/'+ options.username + '?&callback=?';
                cacheId = 'picasa-collection-user-' + options.username;
            }

            // check the cache first
            if( self.options.enableCache ) {
                cacheReq = self.getCache(cacheId);
            }
            else {
                cacheReq = $.Deferred();
                cacheReq.resolve({expired: true});
            }

            cacheReq.done(function(cache) {
                if( !cache.expired ) {
                    configReq.resolve(cache);  // initialize the gallery from cached data
                }
                else {  // cache expired, load fresh data
                    $.ajax({
                        url: picasaUrl,
                        data: {
                            'kind': options.sourceType === 'collection' ? 'album' : 'photo',
                            'access': 'public',
                            'max-results': options.sourceType === 'collection' ? albumnum : options.limit,
                            'thumbsize': options.thumbSize+'u',
                            'imgmax': options.imageSize+'u',
                            'fields': options.sourceType === 'collection' ? 'entry(title,gphoto:name,gphoto:numphotos)': 'entry(gphoto:timestamp,media:group(media:content,media:thumbnail,media:title,media:description))',
                            'alt': 'json-in-script',
                        },
                        dataType: 'jsonp',
                        timeout: 5000,
                        success: function(data) {
                            if( !data.feed.entry ) {
                                self.fgLog('Picasa request failed');
                                return;
                            }

                            var config,
                                getPhotos = [],
                                request;

                            if( options.sourceType === 'collection' ) {
                                config = {albums: []};

                                $.each(data.feed.entry, function(index) {
                                    if( this.gphoto$numphotos.$t === 0 ) {
                                        return;
                                    }

                                    var album = this;

                                    request = $.ajax({
                                        url: 'https://picasaweb.google.com/data/feed/api/user/'+options.username+'/album/'+this.gphoto$name.$t+'?&callback=?',
                                        data: {
                                            'kind': 'photo',
                                            'access': 'public',
                                            'max-results': options.limit,
                                            'thumbsize': options.thumbSize+'u',
                                            'imgmax': options.imageSize+'u',
                                            'fields': 'entry(gphoto:timestamp,media:group(media:content,media:thumbnail,media:title,media:description))',
                                            'alt': 'json-in-script'
                                        },
                                        dataType: 'jsonp',
                                        global: false,
                                        success: function(albumdata) {
                                            if( albumdata.feed.entry ) {
                                                var obj = {};
                                                obj.title = album.title.$t;
                                                obj.items = [];

                                                $.each(albumdata.feed.entry, function(i) {
                                                    obj.items[i] = {};
                                                    if( this.media$group.media$content.length > 1 && (this.media$group.media$content[1].type === 'video/mpeg4' || this.media$group.media$content[1].type === 'video/mp4' || this.media$group.media$content[1].type === 'video/mpeg') ) {
                                                        obj.items[i].type = 'video';
                                                        obj.items[i].mp4 = this.media$group.media$content[1].url;
                                                    }
                                                    else {
                                                        obj.items[i].type = 'photo';
                                                        obj.items[i].source = this.media$group.media$content[0].url;
                                                    }

                                                    obj.sort = {date: parseInt(this.gphoto$timestamp.$t, 10)};
                                                    obj.items[i].thumbnail = this.media$group.media$thumbnail[0].url;
                                                    obj.items[i].title = this.media$group.media$title.$t;
                                                    obj.items[i].description = this.media$group.media$description.$t;
                                                });

                                                config.albums.push(obj);
                                            }
                                        }
                                    });

                                    getPhotos.push(request);
                                });
                            }
                            else {
                                config = [];

                                $.each(data.feed.entry, function(i) {
                                    config[i] = {};

                                    if( this.media$group.media$content.length > 1 && (this.media$group.media$content[1].type === 'video/mpeg4' || this.media$group.media$content[1].type === 'video/mp4' || this.media$group.media$content[1].type === 'video/mpeg') ) {
                                        config[i].type = 'video';
                                        config[i].mp4 = this.media$group.media$content[1].url;
                                    }
                                    else {
                                        config[i].type = 'photo';
                                        config[i].source = this.media$group.media$content[0].url;
                                    }

                                    config[i].sort = {date: parseInt(this.gphoto$timestamp.$t, 10)};
                                    config[i].thumbnail = this.media$group.media$thumbnail[0].url;
                                    config[i].title = this.media$group.media$title.$t;
                                    config[i].description = this.media$group.media$description.$t;
                                });
                            }

                            $.when.apply($, getPhotos).done(function() {
                                self.options.enableCache && self.updateCache(config, cacheId);  // update the cache file with the loaded data
                                configReq.resolve(config);
                            });
                        },
                        complete: function(xhr, data) {
                            var msg = '';
                            if(xhr.status === 0) {
                                if( options.sourceType === 'user' ) {
                                    msg += 'user not found';
                                }
                                else if ( options.sourceType === 'album' ) {
                                    msg += 'user or album not found';
                                }

                                msg = 'Picasa request failed' + (msg ? ': '+msg : '.');

                                self.fgLog(msg);
                            }
                        }
                    });
                }
            });

            return configReq;
        },



        // function to log messages in a cross-browser manner
        fgLog: function() {
            if( window.console && window.console.log ) {
                console.log('[flowGallery] ' + Array.prototype.slice.call(arguments));
            }
            else {
                alert( '[flowGallery] ' + Array.prototype.slice.call(arguments).join('<br>') );
            }
        }

    };  // end FlowPhotoFeed


    // create the jquery plugin
    $.fn.flowGalleryPhotoFeed = function(options) {
        var opts = $.extend( true, {}, $.fn.flowGallery.defaults, $.fn.flowGalleryPhotoFeed.defaults, options );

        // $.extend deep copy merges arrays too, which we don't want
        if( $.isArray(options.sortBy) ) {
            opts.sortBy = options.sortBy;
        }

        return this.each(function () {
            //prevent against multiple instantiations
            if( !$.data(this, 'flowGalleryPhotoFeed') ) {
                var gallery = Object.create(FlowPhotoFeed);
                gallery.init(opts, this);
                $.data(this, 'flowGalleryPhotoFeed', true);
            }
        });
    };


    $.fn.flowGalleryPhotoFeed.defaults = {
        feed: 'flickr',  // flickr, picasa
        sortBy: ['title', 'date'],
        flickrOptions: {
            apiKey: '',
            sourceType: 'text',  // text, tags, user, set, group, collection
            sourceId: '',
            userId: '',
            limit: 30,
            numberOfAlbums: 'all',
            sort: 'relevance',
            thumbSize: 's',  // s, q, t
            imageSize: 'l',  // c, l, o
            media: 'all'  // all, photo, video
        },
        picasaOptions: {
            sourceType: 'search',  // search, user, album, collection
            username: '',
            album: '',
            search: '',
            numberOfAlbums: 'all',
            limit: 30,
            thumbSize: 288,
            imageSize: 1280
        }
    };

}) (jQuery, document, window);