<?php

/**
 * Fichier: SalleController.php
 *
 * Easy-Sports.net
 * Controleur de l'entité salle
 *
 *
 * PHP version 5
 *
 * @package    Easysports
 * @subpacakge Administration section
 * @category   Classes
 * @author     benbet <info@siteweb-service.be>
 * @copyright  2015 Siteweb-Service
 * @link       http://www.easy-sports.net
 * @version    Version 1.0
 */

namespace sws\Bundle\BackBundle\Controller;

use Doctrine\ORM\EntityNotFoundException;
use sws\Bundle\BackBundle\Entity\Salle;
use Symfony\Bundle\FrameworkBundle\Controller\Controller;
use Symfony\Component\HttpFoundation\Request;
use sws\Bundle\BackBundle\Form\Type\SalleType;
use sws\Bundle\BackBundle\Form\Handler\SalleHandler;

class SalleController extends Controller
{

    /**
     * Affiche une entité Salle
     *
     * @param int $id Valeur de l'identification de la salle
     * @param Request $request Contenu de la requête HTTP
     *
     * @return form vers la template swsBackBundle:Salle:view.html.twig
     */
    public function viewAction($id, Request $request)
    {
        $repository = $this->getDoctrine()->getManager()->getRepository('swsBackBundle:Salle');
        $salle = $repository->find($id);

        if (!$salle) {
            throw $this->createNotFoundException("La salle avec l'identifiant ".$id." n'existe pas.");
        }

        $formHandler = new SalleHandler($this->createForm(new SalleType(), $salle, array('disabled' => true)), $request);

        return $this->render('swsBackBundle:Salle:view.html.twig',array('form' => $formHandler->getForm()->createView()));

    }

    /**
     * Supprime une entité Salle
     *
     * @param int $id Valeur de l'identification de la salle
     * @param Request $request Contenu de la requête HTTP
     *
     * @return route sws_back_listsalle
     */
    public function deleteAction($id, Request $request)
    {
        $repository = $this->getDoctrine()->getManager()->getRepository('swsBackBundle:Salle');
        $salle = $repository->find($id);

        if (!$salle) {
            throw $this->createNotFoundException("La salle avec l'identifiant ".$id." n'existe pas.");
        }

        $em = $this->getDoctrine()->getEntityManager();
        $em->remove($salle);
        $em->flush();

        return $this->redirect($this->generateUrl('sws_back_listsalle'));

    }

    /**
     * Crée une entité Salle
     *
     * @param Request $request Contenu de la requête HTTP
     *
     * @return route sws_back_listsalle
     * @return form vers la template swsBackBundle:Salle:create.html.twig
     */
    public function createAction(Request $request)
    {

        $formHandler = new SalleHandler($this->createForm(new SalleType(), new Salle()), $request);

        if ($formHandler->verification()) {

            $salle = $formHandler->getForm()->getData();

            //$joueur->setUser($this->get('security.context')->getToken()->getUser());
            //var_dump($this->get('security.context')->getToken()->getUser()->getemail());

            $salle->setUser($this->getUser());


            $em = $this->getDoctrine()->getEntityManager();
            $em-> persist($salle);
            $em->flush();

            return $this->redirect($this->generateUrl('sws_back_listsalle'));
        }

        return $this->render('swsBackBundle:Salle:create.html.twig',array(
            'form' => $formHandler->getForm()->createView()
            )
        );
    }

    /**
     * Modifie une entité Salle
     *
     * @param int $id Valeur de l'identification de la salle
     * @param Request $request Contenu de la requête HTTP
     *
     * @return route sws_back_listsalle
     * @return form vers la template swsBackBundle:Salle:modify.html.twig
     */
    public function modifyAction($id, Request $request)
    {
        $repository = $this->getDoctrine()->getManager()->getRepository('swsBackBundle:Salle');
        $salle = $repository->find($id);

        if (!$salle) {
            throw $this->createNotFoundException("La salle avec l'identifiant ".$id." n'existe pas.");
        }

        $formHandler = new SalleHandler($this->createForm(new SalleType(), $salle), $request);

        if ($formHandler->verification()) {

            $em = $this->getDoctrine()->getEntityManager();
            $em-> persist($formHandler->getForm()->getData());
            $em->flush();

            return $this->redirect($this->generateUrl('sws_back_listsalle'));
        }

        return $this->render('swsBackBundle:Salle:modify.html.twig',array('form' => $formHandler->getForm()->createView()));
    }

    /**
     * Affiche la liste des salle
     *
     * Si le paramètre $critereParam est = à 'no#critere", la fonction renvoie les joueurs sans appliquer de filtre.
     * Si le paramètre $critereParam est != de 'no#critere", la fonction renvoie les salles filtrées sur le nom et
     * le nom succinct selon le critère fourni.
     *
     * Le paramètre "critereParam" de la route sws_back_listjoueur est initialisé par défaut à 'no#critere' via la
     * configuration de la route.
     *
     * @param int $id Valeur de l'identification de la salle
     * @param Request $request Contenu de la requête HTTP
     *
     * @return route sws_back_listsalle
     * @return form vers la template swsBackBundle:Salle:modify.html.twig
     */
    public function listAction($page, $critereParam, Request $request)
    {
        if (!is_numeric($page)) {
            throw $this->createNotFoundException("Le N° de page est incorrect: ".$page);
        }

        $form = $this->get('form.factory')->createBuilder('form')
            ->add('critere','text')
            ->getForm()
        ;

        $form->handleRequest($request);

        if ($form->isValid() or $critereParam <> 'no#critere') {

            if ($page < 1) {
                throw $this->createNotFoundException("La page ".$page." n'existe pas.");
            }

            $nbPerPage = 15;
            $postForm = $request->request->get('form');

            if ($critereParam <> 'no#critere'){

                $critere = $critereParam;
            }
            else{

                $critere = $postForm['critere'];
            }


            $listSalles = $this->getDoctrine()
                ->getManager()
                ->getRepository('swsBackBundle:Salle')
                ->getSallesByCritere($critere, $page, $nbPerPage)
            ;

            $nbPages = ceil(count($listSalles)/$nbPerPage);

            /*
            if ($page > $nbPages) {
                throw $this->createNotFoundException("La page ".$page." n'existe pas.");
            }
            */

            return $this->render('swsBackBundle:Salle:list.html.twig', array(
                'listSalles' => $listSalles,
                'nbPages'     => $nbPages,
                'page'        => $page,
                'form'        => $form->createView(),
                'critereParam' => $critere
            ));
        }
        else {

            if ($page < 1) {
                throw $this->createNotFoundException("La page ".$page." n'existe pas.");
            }

            $nbPerPage = 15;

            $listSalles = $this->getDoctrine()
                ->getManager()
                ->getRepository('swsBackBundle:Salle')
                ->getSalles($page, $nbPerPage)
            ;

            $nbPages = ceil(count($listSalles)/$nbPerPage);

            /*
            if ($page > $nbPages) {
                throw $this->createNotFoundException("La page ".$page." n'existe pas.");
            }
            */

            return $this->render('swsBackBundle:Salle:list.html.twig', array(
                'listSalles' => $listSalles,
                'nbPages'     => $nbPages,
                'page'        => $page,
                'form'        => $form->createView(),
                'critereParam' => 'no#critere'
            ));
        }
    }

    /**
     * Modifie l'état d'une entité Salle
     *
     * Si l'état de la salle est inactif (= 0) alors l'état est initialisé à actif (=1).
     * Si l'état de la salle est actif (= 1) alors l'état est initialisé à inactif (=0).
     *
     * @param int $id Valeur de l'identification de la coporation
     *
     * @return route sws_back_listsalle
     */
    public function modifyStatusAction($id)
    {
        $repository = $this->getDoctrine()->getManager()->getRepository('swsBackBundle:Salle');
        $salle = $repository->find($id);

        if (!$salle) {
            throw $this->createNotFoundException("La salle avec l'identifiant ".$id." n'existe pas.");
        }

        $status = $salle->getStatus();

        if ($status == 0)
        {
            $salle->setStatus(1);
        }
        else
        {
            $salle->setStatus(0);
        }

        $em = $this->getDoctrine()->getEntityManager();
        $em-> persist($salle);
        $em->flush();

        return $this->redirect($this->generateUrl('sws_back_listsalle'));
    }

    /**
     * Retourne les salles pour un championnat donné
     *
     * Seules les salles n'étant pas encore associées au championnat donné sont retournées.
     *
     * @param int $id Valeur de l'identification du championnat à traiter
     * @param Request $request Contenu de la requête HTTP
     *
     * @return array $listSalles au format json
     */
    public function getSallesChAction($id, Request $request)
    {

            $listSalles = $this->getDoctrine()
                ->getManager()
                ->getRepository('swsBackBundle:Salle')
                ->getSallesCh($id);

        return json_encode($listSalles);


    }

    public function exportXlsAction()
    {
        $now = new \Datetime();
        $now_format = $now->format('d-m-y ');
        $now_format2 = $now->format('d_m_y ');


        $listSalles = $this->getDoctrine()
            ->getManager()
            ->getRepository('swsBackBundle:Salle')
            ->getAll()
        ;

        // Traitement des dates

        $i = 0;

        foreach ($listSalles as $salle){
            $listSalles2[$i] = $salle;
            $listSalles2[$i]['updatedAt'] = $salle['updatedAt']->format('d-m-Y');
            $listSalles2[$i]['createdAt'] = $salle['createdAt']->format('d-m-Y');

            $i = $i + 1;
        }

        // ask the service for a Excel5
        $phpExcelObject = $this->get('phpexcel')->createPHPExcelObject();


        $phpExcelObject->getProperties()->setCreator($this->getUser()->getNom().' '.$this->getUser()->getPrenom())
            ->setTitle("Liste des joueurs de la RILTennis au ".$now_format);
        $phpExcelObject->setActiveSheetIndex(0);
        $phpExcelObject->getActiveSheet()
            ->fromArray(
                $listSalles2,  // The data to set
                NULL,        // Array values with this value will not be set
                'A1'         // Top left coordinate of the worksheet range where
            //    we want to set these values (default is A1)
            );

        $phpExcelObject->getActiveSheet()->setTitle('Liste des salles au '.$now_format);
        // Set active sheet index to the first sheet, so Excel opens this as the first sheet
        $phpExcelObject->setActiveSheetIndex(0);

        // create the writer
        $writer = $this->get('phpexcel')->createWriter($phpExcelObject, 'Excel5');
        // create the response
        $response = $this->get('phpexcel')->createStreamedResponse($writer);
        // adding headers
        $response->headers->set('Content-Type', 'text/vnd.ms-excel; charset=utf-8');

        $fileName = 'RIL_liste_salles_'.$now_format2.'.xls';
        $sanitizer = $this->get('sanitizeFileName');
        $sanitizer->processSanitizeFileNamle($fileName);
        $fileName = $sanitizer->getFileName();

        $response->headers->set('Content-Disposition', 'attachment;filename='.$fileName);
        $response->headers->set('Pragma', 'public');
        $response->headers->set('Cache-Control', 'maxage=1');

        return $response;

    }

}

