<?php

/**
 * Fichier: JoueursController.php
 *
 * Easy-Sports.net
 * Controleur de l'entité joueurs
 *
 *
 * PHP version 5
 *
 * @package    Easysports
 * @subpacakge Administration section
 * @category   Classes
 * @author     benbet <info@siteweb-service.be>
 * @copyright  2015 Siteweb-Service
 * @link       http://www.easy-sports.net
 * @version    Version 1.0
 */

namespace sws\Bundle\BackBundle\Controller;

use Doctrine\ORM\EntityNotFoundException;
use sws\Bundle\BackBundle\Entity\Joueurs;
use Symfony\Bundle\FrameworkBundle\Controller\Controller;
use Symfony\Component\HttpFoundation\Request;
use sws\Bundle\BackBundle\Form\Type\JoueursType;
use sws\Bundle\BackBundle\Form\Handler\JoueursHandler;

class JoueursController extends Controller
{

    /**
     * Affiche une entité Joueur
     *
     * @param int $id Valeur de l'identification du joueur
     * @param Request $request Contenu de la requête HTTP
     *
     * @return form vers la template swsBackBundle:Joueurs:view.html.twig
     */
    public function viewAction($id, Request $request)
    {
        $repository = $this->getDoctrine()->getManager()->getRepository('swsBackBundle:Joueurs');
        $joueur = $repository->find($id);

        if (!$joueur) {
            throw $this->createNotFoundException("Le joueur avec l'identifiant ".$id." n'existe pas.");
        }

        $formHandler = new JoueursHandler($this->createForm(new JoueursType(), $joueur, array('disabled' => true)), $request);

        return $this->render('swsBackBundle:Joueurs:view.html.twig',array('form' => $formHandler->getForm()->createView()));

    }

    /**
     * Supprime une entité Joueur
     *
     * @param int $id Valeur de l'identification du joueur
     * @param Request $request Contenu de la requête HTTP
     *
     * @return route sws_back_listjoueur
     */
    public function deleteAction($id, Request $request)
    {
        $repository = $this->getDoctrine()->getManager()->getRepository('swsBackBundle:Joueurs');
        $joueur = $repository->find($id);

        if (!$joueur) {
            throw $this->createNotFoundException("Le joueur avec l'identifiant ".$id." n'existe pas.");
        }

        $em = $this->getDoctrine()->getEntityManager();
        $em->remove($joueur);
        $em->flush();

        return $this->redirect($this->generateUrl('sws_back_listjoueur'));

    }

    /**
     * Crée une entité Joueur
     *
     * @param Request $request Contenu de la requête HTTP
     *
     * @return route sws_back_listjoueur
     * @return form vers la template swsBackBundle:Joueurs:create.html.twig
     */
    public function createAction(Request $request)
    {

        $formHandler = new JoueursHandler($this->createForm(new JoueursType(), new Joueurs()), $request);

        if ($formHandler->verification()) {

            $joueur = $formHandler->getForm()->getData();


            //$joueur->setUser($this->get('security.context')->getToken()->getUser());
            //var_dump($this->get('security.context')->getToken()->getUser()->getemail());

            $joueur->setUser($this->getUser());
            

            $em = $this->getDoctrine()->getEntityManager();
            $em-> persist($joueur);
            $em->flush();

            return $this->redirect($this->generateUrl('sws_back_listjoueur'));
        }

        return $this->render('swsBackBundle:Joueurs:create.html.twig',array('form' => $formHandler->getForm()->createView()));
    }

    /**
     * Modifie une entité Joueur
     *
     * @param int $id Valeur de l'identification du joueur
     * @param Request $request Contenu de la requête HTTP
     *
     * @return route sws_back_listjoueur
     * @return form vers la template swsBackBundle:Joueurs:modify.html.twig
     */
    public function modifyAction($id, Request $request)
    {
        $repository = $this->getDoctrine()->getManager()->getRepository('swsBackBundle:Joueurs');
        $joueur = $repository->find($id);

        if (!$joueur) {
            throw $this->createNotFoundException("Le joueur avec l'identifiant ".$id." n'existe pas.");
        }

        $formHandler = new JoueursHandler($this->createForm(new JoueursType(), $joueur), $request);

        if ($formHandler->verification()) {

            $em = $this->getDoctrine()->getEntityManager();
            $joueur->setUser($this->getUser());
            $em-> persist($formHandler->getForm()->getData());
            $em->flush();

            return $this->redirect($this->generateUrl('sws_back_listjoueur'));
        }

        return $this->render('swsBackBundle:Joueurs:modify.html.twig',array('form' => $formHandler->getForm()->createView()));
    }

    /**
     * Affiche la liste des joueurs
     *
     * Si le paramètre $critereParam est = à 'no#critere", la fonction renvoie les joueurs sans appliquer de filtre.
     * Si le paramètre $critereParam est != de 'no#critere", la fonction renvoie les joueurs filtrés sur le nom et
     * le prénom selon le critère fourni.
     *
     * Le paramètre "critereParam" de la route sws_back_listjoueur est initialisé par défaut à 'no#critere' via la
     * configuration de la route.
     *
     * @param int $id Valeur de l'identification du joueur
     * @param Request $request Contenu de la requête HTTP
     *
     * @return route sws_back_listjoueur
     * @return form vers la template swsBackBundle:Joueurs:modify.html.twig
     */
    public function listAction($page, $critereParam, Request $request)
    {
        if (!is_numeric($page)) {
            throw $this->createNotFoundException("La n° de page est incorrect: ".$page);
        }

        $form = $this->get('form.factory')->createBuilder('form')
          ->add('critere','text')
          ->getForm()
        ;

        $form->handleRequest($request);

        if ($form->isValid() or $critereParam <> 'no#critere') {
          
           if ($page < 1) {
              throw $this->createNotFoundException("La page ".$page." n'existe pas.");
            }

            $nbPerPage = 15;
            $postForm = $request->request->get('form');

            if ($critereParam <> 'no#critere'){

                $critere = $critereParam;    
            }
            else{
                
                $critere = $postForm['critere'];    
            }
            

            $listJoueurs = $this->getDoctrine()
              ->getManager()
              ->getRepository('swsBackBundle:Joueurs')
              ->getJoueursByCritere($critere, $page, $nbPerPage)
            ;

            $nbPages = ceil(count($listJoueurs)/$nbPerPage);

            /*
            if ($page > $nbPages) {
              throw $this->createNotFoundException("La page ".$page." n'existe pas.");
            }
            */

            return $this->render('swsBackBundle:Joueurs:list.html.twig', array(
                'listJoueurs' => $listJoueurs,
                'nbPages'     => $nbPages,
                'page'        => $page,
                'form'        => $form->createView(),
                'critereParam' => $critere
            ));
        }
        else {

            if ($page < 1) {
              throw $this->createNotFoundException("La page ".$page." n'existe pas.");
            }

            $nbPerPage = 15;

            $listJoueurs = $this->getDoctrine()
              ->getManager()
              ->getRepository('swsBackBundle:Joueurs')
              ->getJoueurs($page, $nbPerPage)
            ;

            $nbPages = ceil(count($listJoueurs)/$nbPerPage);

            /*
            if ($page > $nbPages) {
              throw $this->createNotFoundException("La page ".$page." n'existe pas.");
            }
            */

            return $this->render('swsBackBundle:Joueurs:list.html.twig', array(
                /*'joueurs' => $joueurs,*/
                'listJoueurs' => $listJoueurs,
                'nbPages'     => $nbPages,
                'page'        => $page,
                'form'        => $form->createView(),
                'critereParam' => 'no#critere'
            ));
        }
    }

    /**
     * Modifie l'état d'une entité Joueur
     *
     * Si l'état du joueur est inactif (= 0) alors l'état est initialisé à actif (=1).
     * Si l'état du joueur est actif (= 1) alors l'état est initialisé à inactif (=0).
     *
     * @param int $id Valeur de l'identification du joueur
     *
     * @return route sws_back_listjoueur
     */
   public function modifyStatusAction($id)
    {
        $repository = $this->getDoctrine()->getManager()->getRepository('swsBackBundle:Joueurs');
        $joueur = $repository->find($id);

        if (!$joueur) {
            throw $this->createNotFoundException("Le joueur avec l'identifiant ".$id." n'existe pas.");
        }

        $status = $joueur->getStatus();

       if ($status == 0)
        {
            $joueur->setStatus(1);
        }
        else
        {
            $joueur->setStatus(0);
        }

        $em = $this->getDoctrine()->getEntityManager();
        $em-> persist($joueur);
        $em->flush();

        return $this->redirect($this->generateUrl('sws_back_listjoueur'));
    }

    public function exportXlsAction()
    {
        $now = new \Datetime();
        $now_format = $now->format('d-m-y ');
        $now_format2 = $now->format('d_m_y ');


        $listJoueurs = $this->getDoctrine()
            ->getManager()
            ->getRepository('swsBackBundle:Joueurs')
            ->getAll()
        ;

        // Traitement des dates

        $i = 0;

        foreach ($listJoueurs as $joueur){
            $listJoueurs2[$i] = $joueur;
            $listJoueurs2[$i]['dateNaissance'] = $joueur['dateNaissance']->format('d-m-Y');
            $listJoueurs2[$i]['updatedAt'] = $joueur['updatedAt']->format('d-m-Y');
            $listJoueurs2[$i]['createdAt'] = $joueur['createdAt']->format('d-m-Y');

            $i = $i + 1;
        }

        // ask the service for a Excel5
        $phpExcelObject = $this->get('phpexcel')->createPHPExcelObject();


        $phpExcelObject->getProperties()->setCreator($this->getUser()->getNom().' '.$this->getUser()->getPrenom())
            ->setTitle("Liste des joueurs de la RILTennis au ".$now_format);
        $phpExcelObject->setActiveSheetIndex(0);
        $phpExcelObject->getActiveSheet()
            ->fromArray(
                $listJoueurs2,  // The data to set
                NULL,        // Array values with this value will not be set
                'A1'         // Top left coordinate of the worksheet range where
                //    we want to set these values (default is A1)
            );

        $phpExcelObject->getActiveSheet()->setTitle('Liste des joueurs au '.$now_format);
        // Set active sheet index to the first sheet, so Excel opens this as the first sheet
        $phpExcelObject->setActiveSheetIndex(0);

        // create the writer
        $writer = $this->get('phpexcel')->createWriter($phpExcelObject, 'Excel5');
        // create the response
        $response = $this->get('phpexcel')->createStreamedResponse($writer);
        // adding headers
        $response->headers->set('Content-Type', 'text/vnd.ms-excel; charset=utf-8');

        $fileName = 'RIL_liste_joueurs_'.$now_format2.'.xls';
        $sanitizer = $this->get('sanitizeFileName');
        $sanitizer->processSanitizeFileNamle($fileName);
        $fileName = $sanitizer->getFileName();

        $response->headers->set('Content-Disposition', 'attachment;filename='.$fileName);
        $response->headers->set('Pragma', 'public');
        $response->headers->set('Cache-Control', 'maxage=1');

        return $response;

    }

}

