<?php

/**
 * Fichier: CorporationController.php
 *
 * Easy-Sports.net
 * Controleur de l'entité corporation
 *
 *
 * PHP version 5
 *
 * @package    Easysports
 * @subpacakge Administration section
 * @category   Classes
 * @author     benbet <info@siteweb-service.be>
 * @copyright  2015 Siteweb-Service
 * @link       http://www.easy-sports.net
 * @version    Version 1.0
 */

namespace sws\Bundle\BackBundle\Controller;

use Doctrine\ORM\EntityNotFoundException;
use sws\Bundle\BackBundle\Entity\Corporation;
use Symfony\Bundle\FrameworkBundle\Controller\Controller;
use Symfony\Component\HttpFoundation\Request;
use sws\Bundle\BackBundle\Form\Type\CorporationType;
use sws\Bundle\BackBundle\Form\Handler\CorporationHandler;
use sws\Bundle\BackBundle\Form\Type\CorporationModifyType;

class CorporationController extends Controller
{

    /**
     * Affiche une entité Corporation
     *
     * @param int $id Valeur de l'identification de la corporation
     * @param Request $request Contenu de la requête HTTP
     *
     * @return form vers la template swsBackBundle:Corporation:view.html.twig
     */
    public function viewAction($id, Request $request)
    {
        $repository = $this->getDoctrine()->getManager()->getRepository('swsBackBundle:Corporation');
        $corporation = $repository->find($id);

        if (!$corporation) {
            throw $this->createNotFoundException("La corporation avec l'identifiant ".$id." n'existe pas.");
        }

        $formHandler = new CorporationHandler($this->createForm(new CorporationType(), $corporation, array('disabled' => true)), $request);

        return $this->render('swsBackBundle:Corporation:view.html.twig',array('form' => $formHandler->getForm()->createView()));

    }

    /**
     * Supprime une entité Corporation
     *
     * @param int $id Valeur de l'identification de la corporation
     * @param Request $request Contenu de la requête HTTP
     *
     * @return route sws_back_listcoporation
     */
    public function deleteAction($id, Request $request)
    {
        $repository = $this->getDoctrine()->getManager()->getRepository('swsBackBundle:Corporation');
        $corporation = $repository->find($id);

        if (!$corporation) {
            throw $this->createNotFoundException("La corporation avec l'identifiant ".$id." n'existe pas.");
        }

        $em = $this->getDoctrine()->getEntityManager();
        $em->remove($corporation);
        $em->flush();

        return $this->redirect($this->generateUrl('sws_back_listcorporation'));

    }

    /**
     * Crée une entité Corporation
     *
     * @param Request $request Contenu de la requête HTTP
     *
     * @return route sws_back_listcorporation
     * @return form vers la template swsBackBundle:Corporation:create.html.twig
     */
    public function createAction(Request $request)
    {

        // Récupérer les types de coporations
        $listTypeCorpo = $this->getDoctrine()
            ->getManager()
            ->getRepository('swsBackBundle:TypeCorpo')
            ->getAll()
        ;

        $formHandler = new CorporationHandler($this->createForm(new CorporationType(), new Corporation()), $request);

        if ($formHandler->verification()) {

            $corporation = $formHandler->getForm()->getData();

            //$joueur->setUser($this->get('security.context')->getToken()->getUser());
            //var_dump($this->get('security.context')->getToken()->getUser()->getemail());

            $corporation->setUser($this->getUser());

            $corporation->upload();
            $em = $this->getDoctrine()->getEntityManager();
            $em-> persist($corporation);
            $em->flush();

            return $this->redirect($this->generateUrl('sws_back_listcorporation'));
        }

        return $this->render('swsBackBundle:Corporation:create.html.twig',array(
            'form' => $formHandler->getForm()->createView(),
            'listTypeCorpo' => $listTypeCorpo
            )
        );
    }

    /**
     * Modifie une entité Corporation
     *
     * @param int $id Valeur de l'identification de la corporation
     * @param Request $request Contenu de la requête HTTP
     *
     * @return route sws_back_listcorporation
     * @return form vers la template swsBackBundle:Corporation:modify.html.twig
     */
    public function modifyAction($id, Request $request)
    {
        $repository = $this->getDoctrine()->getManager()->getRepository('swsBackBundle:Corporation');
        $corporation = $repository->find($id);

        $old_path = $corporation->getPath();

        if (!$corporation) {
            throw $this->createNotFoundException("La corporation avec l'identifiant ".$id." n'existe pas.");
        }

        $formHandler = new CorporationHandler($this->createForm(new CorporationModifyType(), $corporation), $request);

        if ($formHandler->verification()) {


            $corporation = $formHandler->getForm()->getData();

            $form = $request->request->get('corporation_form');
            if (array_key_exists('deleteLogo', $form )) {


                // Suppression de la photo
                $corporation->setPath('/bundles/front2/images/logoRil21x18.png');

                if ($old_path <> '/bundles/front2/images/logoRil21x18.png'){

                    // il faut supprimer le fichier également ($old_path)
                    unlink(__DIR__.'/../../../../..'.$old_path);
                }
            }else{

                if ($corporation->file <> null){
                    $corporation->upload();
                }
                else{
                    $corporation->setPath($old_path);
                }

            }

            $em = $this->getDoctrine()->getEntityManager();
            $em-> persist($corporation);
            $em->flush();




        /*
            $em = $this->getDoctrine()->getEntityManager();
            $em-> persist($formHandler->getForm()->getData());
            $em->flush();
        */

            return $this->redirect($this->generateUrl('sws_back_listcorporation'));
        }

        return $this->render('swsBackBundle:Corporation:modify.html.twig',array('form' => $formHandler->getForm()->createView()));
    }

    /**
     * Affiche la liste des corporation
     *
     * Si le paramètre $critereParam est = à 'no#critere", la fonction renvoie les joueurs sans appliquer de filtre.
     * Si le paramètre $critereParam est != de 'no#critere", la fonction renvoie les corporations filtrées sur le nom et
     * le nom succinct selon le critère fourni.
     *
     * Le paramètre "critereParam" de la route sws_back_listjoueur est initialisé par défaut à 'no#critere' via la
     * configuration de la route.
     *
     * @param int $id Valeur de l'identification de la corporation
     * @param Request $request Contenu de la requête HTTP
     *
     * @return route sws_back_listcorporation
     * @return form vers la template swsBackBundle:Corporation:modify.html.twig
     */
    public function listAction($page, $critereParam, Request $request)
    {
        if (!is_numeric($page)) {
            throw $this->createNotFoundException("Le n° de page est incorrect: ".$page);
        }

        $form = $this->get('form.factory')->createBuilder('form')
            ->add('critere','text')
            ->getForm()
        ;

        $form->handleRequest($request);

        if ($form->isValid() or $critereParam <> 'no#critere') {

            if ($page < 1) {
                throw $this->createNotFoundException("La page ".$page." n'existe pas.");
            }

            $nbPerPage = 15;
            $postForm = $request->request->get('form');

            if ($critereParam <> 'no#critere'){

                $critere = $critereParam;
            }
            else{

                $critere = $postForm['critere'];
            }


            $listCorporations = $this->getDoctrine()
                ->getManager()
                ->getRepository('swsBackBundle:Corporation')
                ->getCorporationsByCritere($critere, $page, $nbPerPage)
            ;

            $nbPages = ceil(count($listCorporations)/$nbPerPage);

            /*
            if ($page > $nbPages) {
                throw $this->createNotFoundException("La page ".$page." n'existe pas.");
            }
            */

            return $this->render('swsBackBundle:Corporation:list.html.twig', array(
                'listCorporations' => $listCorporations,
                'nbPages'     => $nbPages,
                'page'        => $page,
                'form'        => $form->createView(),
                'critereParam' => $critere
            ));
        }
        else {

            if ($page < 1) {
                throw $this->createNotFoundException("La page ".$page." n'existe pas.");
            }

            $nbPerPage = 15;

            $listCorporations = $this->getDoctrine()
                ->getManager()
                ->getRepository('swsBackBundle:Corporation')
                ->getCorporations($page, $nbPerPage)
            ;

            $nbPages = ceil(count($listCorporations)/$nbPerPage);

            /*
            if ($page > $nbPages) {
                throw $this->createNotFoundException("La page ".$page." n'existe pas.");
            }
            */

            return $this->render('swsBackBundle:Corporation:list.html.twig', array(
                'listCorporations' => $listCorporations,
                'nbPages'     => $nbPages,
                'page'        => $page,
                'form'        => $form->createView(),
                'critereParam' => 'no#critere'
            ));
        }
    }

    /**
     * Modifie l'état d'une entité Corporation
     *
     * Si l'état de la corporation est inactif (= 0) alors l'état est initialisé à actif (=1).
     * Si l'état de la corporation est actif (= 1) alors l'état est initialisé à inactif (=0).
     *
     * @param int $id Valeur de l'identification de la coporation
     *
     * @return route sws_back_listcorporation
     */
    public function modifyStatusAction($id)
    {
        $repository = $this->getDoctrine()->getManager()->getRepository('swsBackBundle:Corporation');
        $corporation = $repository->find($id);

        if (!$corporation) {
            throw $this->createNotFoundException("La corporation avec l'identifiant ".$id." n'existe pas.");
        }

        $status = $corporation->getStatus();

        if ($status == 0)
        {
            $corporation->setStatus(1);
        }
        else
        {
            $corporation->setStatus(0);
        }

        $em = $this->getDoctrine()->getEntityManager();
        $em-> persist($corporation);
        $em->flush();

        return $this->redirect($this->generateUrl('sws_back_listcorporation'));
    }

    public function exportXlsAction()
    {
        $now = new \Datetime();
        $now_format = $now->format('d-m-y');
        $now_format2 = $now->format('d_m_y');

        $listCorporations = $this->getDoctrine()
            ->getManager()
            ->getRepository('swsBackBundle:Corporation')
            ->getAll()
        ;

        // Traitement des dates

        $i = 0;


        foreach ($listCorporations as $corporation){
            $listCorporation2[$i] = $corporation;
            $listCorporation2[$i]['updatedAt'] = $corporation['updatedAt']->format('d-m-Y');
            $listCorporation2[$i]['createdAt'] = $corporation['createdAt']->format('d-m-Y');

            $i = $i + 1;
        }

        // ask the service for a Excel5
        $phpExcelObject = $this->get('phpexcel')->createPHPExcelObject();

        $phpExcelObject->getProperties()->setCreator($this->getUser()->getNom().' '.$this->getUser()->getPrenom())
            ->setTitle("Liste complète des corporations au ".$now_format);
        $phpExcelObject->setActiveSheetIndex(0);
        $phpExcelObject->getActiveSheet()
            ->fromArray(
                $listCorporation2,  // The data to set
                NULL,        // Array values with this value will not be set
                'A1'         // Top left coordinate of the worksheet range where
            //    we want to set these values (default is A1)
            );
        $phpExcelObject->getActiveSheet()->setTitle('Liste corporations');
        // Set active sheet index to the first sheet, so Excel opens this as the first sheet
        $phpExcelObject->setActiveSheetIndex(0);

        // create the writer
        $writer = $this->get('phpexcel')->createWriter($phpExcelObject, 'Excel5');
        // create the response
        $response = $this->get('phpexcel')->createStreamedResponse($writer);
        // adding headers
        $response->headers->set('Content-Type', 'text/vnd.ms-excel; charset=utf-8');

        $fileName = 'RIL_liste_corporations_'.$now_format2.'.xls';
        $sanitizer = $this->get('sanitizeFileName');
        $sanitizer->processSanitizeFileNamle($fileName);
        $fileName = $sanitizer->getFileName();

        $response->headers->set('Content-Disposition', 'attachment;filename='.$fileName);
        $response->headers->set('Pragma', 'public');
        $response->headers->set('Cache-Control', 'maxage=1');

        return $response;

    }

}

