<?php
/**
 * Created by Eton Digital.
 * User: Milos Milojevic (milos.milojevic@etondigital.com)
 * Date: 5/11/15
 * Time: 10:50 AM
 */

namespace BlogBundle\Controller\Backend;

use Application\Sonata\MediaBundle\Entity\Media;
use Doctrine\Common\Collections\ArrayCollection;
use ED\BlogBundle\Event\ArticleAdministrationEvent;
use ED\BlogBundle\Event\MediaArrayEvent;
use ED\BlogBundle\Event\TaxonomyArrayEvent;
use ED\BlogBundle\Events\EDBlogEvents;
use ED\BlogBundle\Forms\ArticleExcerptType;
use ED\BlogBundle\Forms\ArticlePhotoType;
use ED\BlogBundle\Handler\Pagination;
use ED\BlogBundle\Model\Entity\Article;
use Sensio\Bundle\FrameworkExtraBundle\Configuration\Route;
use Symfony\Component\Finder\Exception\AccessDeniedException;
use Symfony\Component\HttpFoundation\JsonResponse;
use Symfony\Component\HttpFoundation\Request;
use Sensio\Bundle\FrameworkExtraBundle\Configuration\ParamConverter;
use ED\BlogBundle\Controller\Backend\ArticleController as BaseController;

class ArticleController extends BaseController
{
    /**
     * @Route("/article/edit/{slug}", name="ed_blog_admin_article_edit")
     * @ParamConverter("article", class="ED\BlogBundle\Interfaces\Model\ArticleInterface", converter="abstract_converter")
     */
    public function editAction(Request $request, $article)
    {
        $user = $this->getBlogUser();

        if($this->container->get('security.authorization_checker')->isGranted('EDIT_ARTICLE', $article) === false)
        {
            throw new AccessDeniedException('This content is currently unavailable. It may be temporarily unavailable, the link you clicked on may have expired, or you may not have permission to view this page.');
        }

        //get writing lock
        $isWritingLocked = $this->get('ed_blog.handler.article_handler')->isLocked($user, $article);

        //get latest drafted version of the article
        $latestDraft = $this->get('app_repository_article')->findOneBy(array(
            "status" => Article::STATUS_DRAFTED,
            "parent" => $article
        ), array(
            "modifiedAt" => "DESC",
            "id" => "DESC"
        ));

        $draft = $this->get('article_generator')->generateNewDraftFromLatestRevision($latestDraft, $article);
        $mediaManager = $this->container->get('sonata.media.manager.media');
        $dispacher = $this->get('event_dispatcher');
        $dispacher->dispatch(EDBlogEvents::ED_BLOG_ARTICLE_PREUPDATE_INIT, new ArticleAdministrationEvent($article));

        $formMedia = $this->createForm(new ArticlePhotoType());
        $formExcerptMedia = $this->createForm(new ArticleExcerptType());

        $form = $this->createForm('article', $draft);

        if ($request->getMethod() == 'POST')
        {
            $form->handleRequest($request);

            if($form->isValid() && $isWritingLocked === false)
            {
                $em = $this->getDoctrine()->getManager();
                $removedMetaKeys = $form['metaExtras']->getData() ? explode(':', $form['metaExtras']->getData()) : array();

                //force author if no permission to switch blog author
                if($article->getAuthor() != $draft->getAuthor())
                {
                    if($this->isGranted('SWITCH_ARTICLE_AUTHOR'))
                    {
                        //switch author authorized
                        $article->setAuthor($draft->getAuthor());

                        $em->persist($article);
                    }
                    else
                    {
                        //switch back to original author
                        $draft->setAuthor( $article->getAuthor() );
                    }
                }

                if( $this->isGranted('PUBLISH_ARTICLE', $article) && (($article->getStatus() == Article::STATUS_DRAFTED && $form->get('publish')->isClicked()) || $article->getStatus() == Article::STATUS_PUBLISHED && $form->get('update')->isClicked()) )
                {
                    if($this->isGranted('EDIT_PUBLISH_STATUS', $draft) && $article->getStatus() != $draft->getStatus())
                    {
                        //article converted to draft
                        $article->setStatus($draft->getStatus());
                    }
                    else
                    {
                        //new changes should be published or updated for already published articles
                        $this->get('article_generator')->generateArticleFromDraft($article, $draft);

                        foreach($article->getMetaData() as $meta)
                        {
                            if( in_array($meta->getKey(), $removedMetaKeys))
                            {
                                $article->removeMetaData($meta);
                            }
                        }

                        /*$article
                            ->setPublishedAt(new \DateTime());*/
                    }

                    $em->persist($article);
                }

                //save latest draft as a new revision
                if($draft->getTitle() != $latestDraft->getTitle() || $draft->getContent() != $latestDraft->getContent())
                {
                    //save as new revision
                    $draft
                        ->setStatus(Article::STATUS_DRAFTED)
                        ->setSlug('drafted-' . $article->getId());

                    //copy title to latestDraft for updated display in article list
                    $parentDraft = $latestDraft->getParent();
                    $parentDraft
                        ->setTitle($draft->getTitle());

                    $em->persist($parentDraft);
                    $em->persist($draft);
                }
                else
                {
                    //update latest revision
                    //but skip metadata to detect removed
                    $this->get('article_generator')->loadArticle($latestDraft, $draft, true, false);

                    $latestDraft
                        ->setStatus(Article::STATUS_DRAFTED);

                    //removing deleted metaData manually
                    foreach($latestDraft->getMetaData() as $meta)
                    {
                        if( in_array($meta->getKey(), $removedMetaKeys))
                        {
                            $latestDraft->removeMetaData($meta);
                        }
                    }

                    $em->persist($latestDraft);
                }

                $em->flush();

                //note - must be after flush, depends on DB
                $dispacher->dispatch(EDBlogEvents::ED_BLOG_ARTICLE_POST_UPDATE, new ArticleAdministrationEvent($article));

                return $this->redirectToRoute('ed_blog_admin_article_edit', array('slug' => $article->getSlug()));
            }
        }

        return $this->render("EDBlogBundle:Article:create.html.twig",
            array(
                'form' => $form->createView(),
                'form_media' => $formMedia->createView(),
                'form_excerpt_media' => $formExcerptMedia->createView(),
                'medias' => $mediaManager->findAll(),
                'article' => $draft,
                'isLocked' => $isWritingLocked,
                'lockedBy' => ($isWritingLocked === false) ? null : $this->getDoctrine()->getRepository(get_class($user))->findOneBy(array('id' => $isWritingLocked))
            ));
    }
}